-- Phase 3: Module Blueprint Engine + Self-Upgrade System

-- Module Blueprints table
CREATE TABLE IF NOT EXISTS public.sf_module_blueprints (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  created_at TIMESTAMPTZ DEFAULT now(),
  updated_at TIMESTAMPTZ DEFAULT now(),
  created_by UUID NOT NULL,
  name TEXT NOT NULL,
  slug TEXT NOT NULL UNIQUE,
  description TEXT,
  natural_description TEXT,
  
  -- Blueprint structure
  entities JSONB DEFAULT '[]'::jsonb,
  relationships JSONB DEFAULT '[]'::jsonb,
  ui_pages JSONB DEFAULT '[]'::jsonb,
  api_routes JSONB DEFAULT '[]'::jsonb,
  
  -- Tech stack
  backend_stack TEXT DEFAULT 'laravel',
  frontend_stack TEXT DEFAULT 'react',
  
  -- Configuration
  permissions JSONB DEFAULT '[]'::jsonb,
  menu_items JSONB DEFAULT '[]'::jsonb,
  complexity TEXT DEFAULT 'medium',
  
  -- Generation
  status TEXT DEFAULT 'draft',
  generated_files JSONB DEFAULT '{}'::jsonb,
  generation_log TEXT,
  
  -- Metadata
  version TEXT DEFAULT '1.0.0',
  tags TEXT[]
);

-- System Versions table (for self-upgrade tracking)
CREATE TABLE IF NOT EXISTS public.sf_system_versions (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  created_at TIMESTAMPTZ DEFAULT now(),
  version TEXT NOT NULL,
  upgrade_type TEXT NOT NULL,
  description TEXT,
  
  -- Upgrade details
  affected_files JSONB DEFAULT '[]'::jsonb,
  patch_content TEXT,
  diff_preview TEXT,
  
  -- Status
  status TEXT DEFAULT 'pending',
  applied_at TIMESTAMPTZ,
  applied_by UUID,
  
  -- Metadata
  detection_method TEXT,
  optimization_score INTEGER,
  rollback_available BOOLEAN DEFAULT true
);

-- Ensure sf_modules table exists with proper structure
CREATE TABLE IF NOT EXISTS public.sf_modules (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  created_at TIMESTAMPTZ DEFAULT now(),
  updated_at TIMESTAMPTZ DEFAULT now(),
  
  slug TEXT NOT NULL UNIQUE,
  name TEXT NOT NULL,
  description TEXT,
  icon TEXT DEFAULT 'Package',
  
  -- Routing
  route TEXT NOT NULL,
  frontend_entry TEXT,
  
  -- Status
  is_active BOOLEAN DEFAULT true,
  is_system BOOLEAN DEFAULT false,
  
  -- Blueprint reference
  blueprint_id UUID REFERENCES public.sf_module_blueprints(id) ON DELETE SET NULL,
  
  -- Metadata
  category TEXT DEFAULT 'Custom',
  version TEXT DEFAULT '1.0.0',
  author TEXT,
  tags TEXT[]
);

-- Enable RLS
ALTER TABLE public.sf_module_blueprints ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.sf_system_versions ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.sf_modules ENABLE ROW LEVEL SECURITY;

-- RLS Policies for sf_module_blueprints
CREATE POLICY "Owners can view all blueprints"
  ON public.sf_module_blueprints FOR SELECT
  USING (has_role(auth.uid(), 'owner'::app_role));

CREATE POLICY "Owners can create blueprints"
  ON public.sf_module_blueprints FOR INSERT
  WITH CHECK (has_role(auth.uid(), 'owner'::app_role) AND auth.uid() = created_by);

CREATE POLICY "Owners can update blueprints"
  ON public.sf_module_blueprints FOR UPDATE
  USING (has_role(auth.uid(), 'owner'::app_role));

CREATE POLICY "Owners can delete blueprints"
  ON public.sf_module_blueprints FOR DELETE
  USING (has_role(auth.uid(), 'owner'::app_role));

-- RLS Policies for sf_system_versions
CREATE POLICY "Owners can view system versions"
  ON public.sf_system_versions FOR SELECT
  USING (has_role(auth.uid(), 'owner'::app_role));

CREATE POLICY "Owners can create system versions"
  ON public.sf_system_versions FOR INSERT
  WITH CHECK (has_role(auth.uid(), 'owner'::app_role));

CREATE POLICY "Owners can update system versions"
  ON public.sf_system_versions FOR UPDATE
  USING (has_role(auth.uid(), 'owner'::app_role));

-- RLS Policies for sf_modules
CREATE POLICY "Everyone can view active modules"
  ON public.sf_modules FOR SELECT
  USING (is_active = true OR has_role(auth.uid(), 'owner'::app_role));

CREATE POLICY "Owners can manage modules"
  ON public.sf_modules FOR ALL
  USING (has_role(auth.uid(), 'owner'::app_role));

-- Indexes for performance
CREATE INDEX idx_blueprints_slug ON public.sf_module_blueprints(slug);
CREATE INDEX idx_blueprints_status ON public.sf_module_blueprints(status);
CREATE INDEX idx_blueprints_created_by ON public.sf_module_blueprints(created_by);
CREATE INDEX idx_system_versions_status ON public.sf_system_versions(status);
CREATE INDEX idx_modules_slug ON public.sf_modules(slug);
CREATE INDEX idx_modules_route ON public.sf_modules(route);
CREATE INDEX idx_modules_active ON public.sf_modules(is_active);

-- Trigger for updated_at
CREATE TRIGGER update_blueprints_updated_at
  BEFORE UPDATE ON public.sf_module_blueprints
  FOR EACH ROW
  EXECUTE FUNCTION public.update_updated_at();

CREATE TRIGGER update_modules_updated_at
  BEFORE UPDATE ON public.sf_modules
  FOR EACH ROW
  EXECUTE FUNCTION public.update_updated_at();