-- Create SECURITY DEFINER function for safe wallet transfers
-- This allows authenticated users to transfer money while respecting RLS
CREATE OR REPLACE FUNCTION public.execute_wallet_transfer(
  _sender_id uuid,
  _recipient_id uuid,
  _amount numeric,
  _note text DEFAULT NULL
)
RETURNS jsonb
LANGUAGE plpgsql
SECURITY DEFINER
SET search_path = public
AS $$
DECLARE
  _sender_balance numeric;
  _recipient_balance numeric;
  _transaction_id uuid;
BEGIN
  -- Critical: Verify caller is the sender (prevents unauthorized transfers)
  IF auth.uid() != _sender_id THEN
    RAISE EXCEPTION 'Unauthorized: You can only transfer from your own wallet';
  END IF;
  
  -- Prevent self-transfers
  IF _sender_id = _recipient_id THEN
    RAISE EXCEPTION 'Cannot transfer to yourself';
  END IF;
  
  -- Validate amount
  IF _amount <= 0 THEN
    RAISE EXCEPTION 'Transfer amount must be positive';
  END IF;
  
  -- Check sender balance and lock row for atomic update
  SELECT balance INTO _sender_balance
  FROM wallets
  WHERE user_id = _sender_id
  FOR UPDATE;
  
  IF NOT FOUND THEN
    RAISE EXCEPTION 'Sender wallet not found';
  END IF;
  
  IF _sender_balance < _amount THEN
    RAISE EXCEPTION 'Insufficient balance';
  END IF;
  
  -- Deduct from sender
  UPDATE wallets
  SET balance = balance - _amount
  WHERE user_id = _sender_id;
  
  -- Add to recipient (create wallet if doesn't exist)
  INSERT INTO wallets (user_id, balance, currency)
  VALUES (_recipient_id, _amount, 'BDT')
  ON CONFLICT (user_id) DO UPDATE
  SET balance = wallets.balance + _amount;
  
  -- Create transaction record
  INSERT INTO transactions (sender_id, recipient_id, amount, type, status, note)
  VALUES (_sender_id, _recipient_id, _amount, 'transfer', 'completed', _note)
  RETURNING id INTO _transaction_id;
  
  -- Log success
  INSERT INTO core_brain_logs (type, message, details)
  VALUES (
    'info',
    'Wallet transfer completed',
    jsonb_build_object(
      'sender_id', _sender_id,
      'recipient_id', _recipient_id,
      'amount', _amount,
      'transaction_id', _transaction_id
    )
  );
  
  RETURN jsonb_build_object(
    'success', true,
    'transaction_id', _transaction_id,
    'amount', _amount
  );
EXCEPTION
  WHEN OTHERS THEN
    -- Log error
    INSERT INTO core_brain_logs (type, message, details)
    VALUES (
      'error',
      'Wallet transfer failed: ' || SQLERRM,
      jsonb_build_object(
        'sender_id', _sender_id,
        'recipient_id', _recipient_id,
        'amount', _amount,
        'error', SQLERRM
      )
    );
    RAISE;
END;
$$;